//=============================================================================
//
// StopGo.c :
//
// Example program for an encoder motor on M1. The counter input C1 is read
// while the motor runs and as the counter value reaches 1000 the program 
// stops. The motor is started by a switch on input I8 (=1) and stopped on
// release of the switch (=0).
//
//-----------------------------------------------------------------------------
// Disclaimer - Exclusion of Liability
//
// This software is distributed in the hope that it will be useful,but WITHOUT 
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
// FITNESS FOR A PARTICULAR PURPOSE. It can be used an modified by anyone
// free of any license obligations or authoring rights.
//=============================================================================

#include "StdAfx.h"

enum MotorIdx {MOTOR_1 = 0, MOTOR_2, MOTOR_3, MOTOR_4};
enum InpIdx {I1=0, I2, I3, I4, I5, I6, I7, I8};
enum CntIdx {COUNT_1 = 0, COUNT_2, COUNT_3, COUNT_4};

#define MOTOR_ON    TRUE
#define MOTOR_OFF   FALSE

#define STRMAXLEN   80

HANDLE  fthdl;
DWORD   errCode;
char    ComPortName[STRMAXLEN];
char    LibVersion[STRMAXLEN];


//=============================================================================

/*-----------------------------------------------------------------------------
 *  MotorStopGo  
 *---------------------------------------------------------------------------*/
static void MotorStopGo(void) {

    INT16   value, lastval;
    INT16   count, cntmode;
    BOOL32  overrun;
    int     duty;

    //  set Motor output ON, stop motor
    SetFtMotorConfig(fthdl, TA_LOCAL, MOTOR_1, MOTOR_ON);
    SetOutMotorValues(fthdl, TA_LOCAL, MOTOR_1, 0, 0);

    //  set Input I8 to 'Digital 5kOhm' mode
    SetFtUniConfig(fthdl, TA_LOCAL, I8, MODE_R, TRUE);

    //  reset counter 1 (index=0)
    printf("reset counter 1...");
    StartCounterReset(fthdl, TA_LOCAL, COUNT_1);

    //  waiting for counter reset
    do {
        GetInCounterValue(fthdl, TA_LOCAL, COUNT_1, &count, &cntmode);
    } while (count > 0);
    printf("OK\n");


    //  init value
    lastval = 0;

    while (1) {

        //  read counter value from Transfer Area
        GetInCounterValue(fthdl, TA_LOCAL, COUNT_1, &count, &cntmode);

        if (count >= 1000) break;

        //  read input value from Transfer Area
        GetInIOValue(fthdl, TA_LOCAL, I8, &value, &overrun);

        if (lastval == value) continue;

        printf("Counter 1 = %5d\n", count);

        //  set motor values, run (= 1) or stop (= 0) motor
        duty = (value) ? DUTY_MAX : 0;
        SetOutMotorValues(fthdl, TA_LOCAL, MOTOR_1, duty, 0);

        lastval = value;
    }

    //  stop motor, distance is reached
    SetOutMotorValues(fthdl, TA_LOCAL, MOTOR_1, 0, 0);

    //  actual counter value
    printf("stop Motor, distance is reached (count= %d) ...\n", count);
}

/*-----------------------------------------------------------------------------
 *  CheckParameter  
 *---------------------------------------------------------------------------*/
static int CheckCOMPar(int argc, char *argv[]) {

    char    *pStr;
    int     comNo;

    if (argc >= 2) {
        if (strlen(argv[1]) > 3) {
            if ((pStr=strstr(argv[1],"COM")) != NULL) {
                sscanf(pStr+3, "%d", &comNo);
                if (comNo >= 1 && comNo <= 255)
                    return 0;
                else {
                    printf("StopGo.exe: invalid COM number...\n\n");
                    return 1;
                }
            }
        }
    }

    printf("StopGo.exe: no input given...\n\n");
    return 1;
}

/*-----------------------------------------------------------------------------
 *  main
 *  
 *---------------------------------------------------------------------------*/
int main(int argc, char *argv[]) {

    printf("\nExample StopGo.exe ...\n");

    //  check input paramter
    if (CheckCOMPar(argc,argv)) {
        printf("Usage: StopGo.exe COMxx\t(e.g. COM2 or COM32)\n");
        return 1;
    }

    //  get library version
    ftxGetLibVersionStr(LibVersion, STRMAXLEN);
    printf("\nftMscLib %s\n", LibVersion);

    //  library initialization
    errCode = ftxInitLib();

    strcpy(ComPortName, argv[1]);
    printf("\n\nOpen ComPort '%s' ...\n", ComPortName);

    //  open COM port
    fthdl = ftxOpenComDevice(ComPortName, 38400, &errCode);

    if (errCode == FTLIB_ERR_SUCCESS) {

        printf("Connected to ROBO TX Controller ...\n");

        //  starting Transfer Area
        errCode = ftxStartTransferArea(fthdl);

        if (errCode == FTLIB_ERR_SUCCESS) {

            printf("Transfer Area was started and runs...\n");

            MotorStopGo();

            //  stop Transfer Area
            ftxStopTransferArea(fthdl);
        }

        else {
            //  error case
            printf("Error: Transfer Area was not started !\n");
        }

        //  closing port
        printf("Closing ComPort '%s' ...\n", ComPortName);
        errCode = ftxCloseDevice(fthdl);
    }

    else {
        //  error case
        printf("Error: No interface available (Port '%s')\n", ComPortName);
    }

    //  close library
    ftxCloseLib();

    return 0;
}
